// Tagging functionality for the Reading List and Reading View
rdb.tagging = function () {
    var userTagLimit = 500,
        tagList = [],
        // This is an array that only contains the tag text for each tag. We
        // need it for the autocomplete
        autocompleteTagList = [],
        tagListEl = $('nav.tags ul'),
        tagInListTemplate = $('#tag-in-nav').html(),
        isReadingList = $('body').attr('id') === 'reading-list';

    // Show/hide for all tags in the list
    function listControl () {
        var trigger = $('a.tags-show-all'),
            list = $('nav.tags ul'),
            limitClass = 'limit-tags';

        if (list.find('li').length > 10) {
            trigger.removeClass('hidden');
        }
        else {
            trigger.addClass('hidden');
        }

        function toggleDisplay (e) {
            e.preventDefault();

            list.toggleClass(limitClass);
            trigger.find('span').toggleClass('hidden');

            rdb.setKeyValue({'expand_tag_list': !list.hasClass(limitClass)});
        }

        $('nav.tags').unbind('click.show-hide');
        $('nav.tags').on('click.show-hide', 'a.tags-show-all', toggleDisplay);
    }

    function updateTagCounts () {
        // Dirty update of number of tags
        $('h1.nav-tags b').text(tagList.length);
        $('a.tags-show-all span b').text(tagList.length);
    }

    function renderTagList () {
        var html = '';

        _.each(tagList, function (tag) {
            html += _.template(tagInListTemplate, tag);
        });

        tagListEl.html(html);

        if (tagList.length > 0) {
            tagListEl.removeClass('hidden');
        }
        else {
            tagListEl.addClass('hidden');
        }

        listControl();
        updateTagCounts();

        // Bit of an edge case, but if you have the delete confirmation open
        // then you take some other action the highlights will get stuck.
        $('ul.bookmarks li.highlight').removeClass('highlight');
    }

    // Fetch all the user's tags from the server and pass them to callback()
    function populateTagList (callback) {
        $.ajax({
            type: 'GET',
            url: '/tagging/tagcloud/ajax',
            success: function (res) {
                if ($.isFunction(callback)) {
                    callback(res);
                }
            },
            error: function (jqhxr) {
                console.log(jqhxr);
            }
        });
    }

    function updateAutocompleteTagList (list) {
        autocompleteTagList = _.map(list, function (tag) {
            // We need to decode any html entities
            // http://stackoverflow.com/questions/5796718/html-entity-decode
            return $('<div/>').html(tag.tag_text).text();
        });
    }

    // Sort the user's tags first by applied count then alphabetically
    function sortTagList () {
        // Create groups of tags that have the same applied count
        var tagGroups = _.groupBy(tagList, 'applied_count');

        // Sort those tag groups in descending order (5, 4, 3, 2, 1)
        tagGroups = _.sortBy(tagGroups, function (group, key) {
            return -key;
        });

        // In each group of tags with the same count, sort those alphabetically
        // based on the text of the tag
        var groupMap = _.map(tagGroups, function (group) {
            return _.sortBy(group, function (item) {
                return item.tag_text;
            });
        });

        // During grouping and sorting we've created some extra layers that
        // we don't want in the tagList, remove those here.
        tagList = _.flatten(groupMap);
    }

    // Modifications should be an array of tag objects
    function updateTagList (modifications) {
        // We want to make sure modifications is an array. Sometimes we'll
        // just get a single object.
        if (!modifications.length) {
            modifications = [modifications];
        }

        _.each(modifications, function (tag) {
            var tagText = tag.tag_text;

            // TODO: Probably want to check on the tag_id instead of the text
            var existingTag = _.find(tagList, function (item) {
                return item.tag_text === tagText;
            });

            if (existingTag) {
                if (tag.applied_count === 0) {
                    tagList.splice(_.indexOf(tagList, existingTag), 1);
                }
                else {
                    $.extend(existingTag, tag);
                }
            }
            else {
                tagList.push(tag);
            }
        });

        // This modifies tagList
        sortTagList();

        updateAutocompleteTagList(tagList);

        if (isReadingList) {
            renderTagList();
        }
    }

    // el should be a jquery element whose data-tag-count attribute will be update
    function updateBookmarkTagCount (el, increaseOrDecrease) {
        var count = parseInt(el.attr('data-tag-count'), 10),
            methods = {
                'increase': function () {
                    return count += 1;
                },
                'decrease': function () {
                    return count -=1;
                }
            };

        var newCount = methods[increaseOrDecrease]();
        el.attr('data-tag-count', newCount);
        return newCount;
    }

    // Enable/disble the add form depending on the number of tags on the bookmark
    function toggleFormState (flyout, bookmarkLimitExceeded) {
        var userLimitExceeded = tagList.length >= userTagLimit,
            msgs = flyout.find('ul.field-messages'),
            formClass = 'over-limit';

        if (userLimitExceeded) {
            formClass = 'user-limit';
        }

        // Disable or enable form based on the tag count
        if (bookmarkLimitExceeded || userLimitExceeded) {
            flyout.find('form').addClass('disabled ' + formClass);
            flyout.find('input').attr('disabled', 'disabled');
            flyout.find('input[type=text]').val('').blur();

            // TODO: This isn't working correctly on the reading View
            msgs.find('li').addClass('hidden');
            msgs.find('.' + formClass).removeClass('hidden');
        }
        else {
            flyout.find('form').removeClass('disabled ' + formClass);
            flyout.find('input').removeAttr('disabled');

            msgs.find('li').addClass('hidden');
            msgs.find('.info').removeClass('hidden');
        }
    }

    // TODO: This is poorly named
    function adding () {
        var template = $('#tag-in-list').html();

        function submitHandler (e) {
            var target = $(e.target),
                bookmarkId = target.attr('data-bookmark-id'),
                input = target.find('input[type=text]'),
                bookmarkEl = null,
                tagEl = null,
                msgs = target.find('ul.field-messages'),
                val = $.trim(input.val().replace(/,+$/, ''));

            e.preventDefault();

            // Depending on where we're adding from the bookmarkEl needs to differ
            if (isReadingList) {
                url = target.attr('action').replace('0', bookmarkId);
                bookmarkEl = $('li[data-bookmark-id=' + bookmarkId + ']');
                tagEl = bookmarkEl.find('ul.tags-list');
            }
            else {
                url = target.attr('action');
                bookmarkEl = $('article.hentry');
                tagEl = $('ul.tags-list');
            }

            if (val.length > 0) {
                $.ajax({
                    type: 'POST',
                    // Check the template for a comment on this.
                    url: url,
                    data: {
                        'tag_string': val
                    },
                    success: function (tags) {
                        if (tags.length) {
                            var html = '',
                                data = {
                                    bookmark_id: bookmarkId
                                },
                                newTagCount = 0;

                            _.each(tags, function (tag) {
                                $.extend(data, tag);

                                html += _.template(template, data);
                                newTagCount = updateBookmarkTagCount(bookmarkEl, 'increase');
                            });

                            tagEl.removeClass('hidden').append(html);

                            // Update the overall tag list
                            updateTagList(tags);

                            toggleFormState($('.flyout-add-tags'), (newTagCount >= 20));
                        }

                        if (isReadingList) {
                            $('body').trigger('flyout.close');
                        }

                        input.val('');
                    },
                    error: function (jqhxr) {
                        msgs.find('li').addClass('hidden');
                    },
                    statusCode: {
                        400: function (jqhxr) {
                            var res = JSON.parse(jqhxr.responseText),
                                errors = {
                                    'bookmark_limit': function () {
                                        msgs.find('.over-limit').removeClass('hidden');
                                    },
                                    'request_limit': function () {
                                        msgs.find('.request-limit').removeClass('hidden');
                                    },
                                    'user_limit': function () {
                                        msgs.find('.user-limit').removeClass('hidden');
                                    }
                                };

                            var err = errors[res.error];

                            if ($.isFunction(err)) {
                                err();
                            }
                            else {
                                msgs.find('.error.general').removeClass('hidden');
                            }
                        },
                        500: function (jqhxr) {
                            msgs.find('.error.general').removeClass('hidden');
                        }
                    }
                });
            }
            else {
                msgs.find('li').addClass('hidden');
                msgs.find('.invalid').removeClass('hidden');
            }
        }

        $('form.tags-add-form').on('submit', submitHandler);
    }

    // Invoking the poorly named function
    adding();

    function tagAutocomplete (tagList) {
        var el = $('input[name=tags]');

        // Pulled pretty much all of this from the jqueryui example:
        // http://jqueryui.com/autocomplete/#multiple
        function split (val) {
            return val.split(/,\s*/);
        }

        function extractLast (term) {
            return split(term).pop();
        }

        el.bind('keydown', function (e) {
            // don't navigate away from the field on tab when selecting an item
            if (e.keyCode === $.ui.keyCode.TAB &&
                    $(this).data('autocomplete').menu.active) {
                e.preventDefault();
            }
        }).autocomplete({
            minLength: 0,
            source: function (request, response) {
                // delegate back to autocomplete, but extract the last term
                response($.ui.autocomplete.filter(autocompleteTagList, extractLast(request.term)));
            },
            focus: function () {
                // prevent value inserted on focus
                return false;
            },
            select: function(e, ui) {
                var terms = split(this.value);
                terms.pop();
                terms.push(ui.item.value);
                // add placeholder to get the comma-and-space at the end
                terms.push('');
                this.value = terms.join(', ');
                return false;
            }
        });

        // We kept seeing the autocomplete list pop up in the upper left of the
        // screen after it had been closed. I think it was because it still had
        // keyboard controls bound to it. Maybe? We disable the autocomplete
        // on success of adding tags.
        el.bind('focus', function (e) {
            $(this).autocomplete('enable');
        });
    }

    // We're using flyout from behaviors.js to display the flyout, but we
    // need the id to be dynamic so we can use the same markup for multiple
    // bookmarks
    function readingListFlyoutPreappend (e) {
        var flyout = $('div.flyout-add-tags'),
            bookmarkId = $(e.target).attr('data-bookmark-id'),
            updateAttrs = function () {
                flyout.attr('id', 'flyout-add-tags-' + bookmarkId);
                flyout.attr('data-bookmark-id', bookmarkId);
            };

        if (flyout.attr('data-bookmark-id')) {
            if (bookmarkId !== flyout.attr('data-bookmark-id')) {
                flyout.addClass('hidden');
                updateAttrs();
            }
        }
        else {
            updateAttrs();
        }
    }

    // In the reading view, the add tags flyout has two triggers. One in the
    // toolbar and one in the actions at the bottom of the article. Each has
    // a different position and should display slightly different things.
    function readingViewFlyoutPreappend (e) {
        var target = $(e.target),
            flyoutSelector = target.attr('data-flyout-selector').split('#')[1],
            flyout = $('div.flyout-add-tags');

        function updateAttrs () {
            flyout.attr('id', flyoutSelector);

            if (target.hasClass('actions-tags')) {
                flyout.removeClass('tip-left')
                    .addClass('tip-bottom');
            }
            else {
                flyout.removeClass('tip-bottom').addClass('tip-left');
            }
        }

        $('a.article-tags').removeClass('active');

        // If the flyout has an id we know it has been opened at least once by
        // either trigger.
        if (flyout.attr('id')) {
            // Allow the flyout to be closed if it's trigger is clicked again
            // while it is open
            if (flyoutSelector !== flyout.attr('id')) {
                flyout.addClass('hidden');
            }
        }

        updateAttrs();
    }

    // When the tag flyout opens this will be triggered.
    function tagFlyoutOpened (e, target, targetFlyout) {
        if (targetFlyout.hasClass('flyout-add-tags')) {
            var targetPos = target.offset(),
                bookmarkLimitExceeded = false;

            if (isReadingList) {
                // targetFlyout.css({
                //     'top': targetPos.top - targetFlyout.parent().offset().top + target.outerHeight()
                // });

                targetFlyout.find('form')
                    .attr('data-bookmark-id', target.attr('data-bookmark-id'));
            }
            else {
                // Special considerations when opening the tag flyout from the
                // actions at the bottom of an article
                if ($(target).hasClass('actions-tags')) {
                    targetFlyout.css({
                        'position': 'absolute',
                        'top': targetPos.top - (target.height() + targetFlyout.height() * 0.9),
                        'left': targetPos.left - targetFlyout.width() * 0.5
                    });
                }
            }

            if (target.parents('li.bookmark').attr('data-tag-count') >= 20 ||
                $('article.hentry').attr('data-tag-count') >= 20) {

                bookmarkLimitExceeded = true;
            }

            toggleFormState(targetFlyout, bookmarkLimitExceeded);
        }
    }

    function tagFlyoutClosed () {
        // Remove and disable the autocomplete list or it will
        // sometimes just pop up on the screen.
        $('input[name=tags]').autocomplete('close').autocomplete('disable');
    }

    function deleteTagFromBookmark (e) {
        var target = $(e.target),
            bookmarkEl = null;

        // Depending on where we're adding from the bookmarkEl needs to differ
        if (isReadingList) {
            bookmarkEl = target.parents('li.bookmark');
        }
        else {
            bookmarkEl = $('article.hentry');
        }

        e.preventDefault();

        $.ajax({
            type: 'DELETE',
            url: target.attr('href'),
            success: function (res) {

                // On the reading list we need to isolate removal of the tag
                // to the bookmark it was removed from.
                if (isReadingList) {
                    target.parent().remove();
                }
                else {
                    $('li[data-tag-id=' + res.tag_id + ']').remove();
                }

                updateTagList(res);

                var newTagCount = updateBookmarkTagCount(bookmarkEl, 'decrease');

                if (!isReadingList) {
                    toggleFormState($('.flyout-add-tags'), (newTagCount >= 20));

                    if (newTagCount === 0) {
                        $('ul.tags-list').addClass('hidden');
                    }
                }
            },
            error: function (jqhxr) {
                console.log(jqhxr);
            }
        });
    }

    // Handle showing the delete confirmation and the delete request
    function deleting () {
        var tagId = null,
            url = null;

        // Abstracted delete success function. We do this so we can use
        // this for both the success and 404 error.
        function handleDelete (res) {
            var tags = $('li[data-tag-id=' + tagId + ']');

            // We need to update the tag-count for each bookmark that
            // has a tag removed because it was deleted
            if (isReadingList) {
                _.each($('ul.bookmarks').find(tags), function (tag) {
                    var el = $(tag).parents('li.bookmark');
                    updateBookmarkTagCount(el, 'decrease');
                });
            }

            tags.remove();

            var deletedTag = _.find(tagList, function (tag) {
                return tag.tag_id === parseInt(tagId, 10);
            });

            tagList.splice(_.indexOf(tagList, deletedTag), 1);

            updateTagCounts();
            listControl();
        }

        function deleteTag () {
            $.ajax({
                type: 'DELETE',
                url: url,
                success: handleDelete,
                statusCode: {
                    404: handleDelete
                }
            });
        }

        $('nav.tags').on('click', 'a.tag-delete', function (e) {
            e.preventDefault();

            var tag = $(e.target).parents('li');

            tagId = tag.attr('data-tag-id');
            url = $(e.target).attr('href');

            var tagsInList = $('ul.bookmarks li[data-tag-id=' + tagId + ']');
            var confirmation = tag.find('div.delete-confirmation');

            tag.addClass('delete-pending');
            confirmation.removeClass('hidden');
            tagsInList.addClass('highlight');

            // Confirmation yes, no buttons
            // Binding the events here only when they are displayed is a bit
            // more efficient...I think
            confirmation.unbind('click');
            confirmation.on('click', 'button', function (e) {
                var answer = $(this).attr('data-answer');

                if (answer === 'yes') {
                    deleteTag();
                }
                else {
                    tag.removeClass('delete-pending');
                    confirmation.addClass('hidden');
                    tagsInList.removeClass('highlight');
                }
            });
        });
    }

    deleting();

    function init () {
        listControl();

        populateTagList(function (list) {
            tagList = list;
            updateAutocompleteTagList(list);
            tagAutocomplete();
        });

        if (isReadingList) {
            $('ul.bookmarks').on('click', 'a.tag-delete', deleteTagFromBookmark)
                .on('click', 'a.tags-add', readingListFlyoutPreappend);
        }
        else {
            $('a.article-tags').on('click', readingViewFlyoutPreappend);
            $('ul.tags-list').on('click', 'a.tag-delete', deleteTagFromBookmark);
        }

        // When a bookmark is deleted this will be triggered.
        // data should be an object that contains bookmarkId and tags-an array
        // of tag ids that were applied to that bookmark.
        $('body').on('bookmark.deleted', function (e, data) {
            var modifiedTags = _.map(data.tags, function (tagId) {
                var tags = _.find(tagList, function (tag) {
                    var updatedTag;

                    if (tag.tag_id === parseInt(tagId, 10)) {
                        tag.applied_count = tag.applied_count - 1;

                        return tag;
                    }
                });

                return tags;
            });

            updateTagList(modifiedTags);
        });

        $('body').on('opened.flyout', tagFlyoutOpened);
        $('body').on('closed.flyout', tagFlyoutClosed);
    }

    init();
}();