// Create the readability namespace.
var readability = (typeof readability !== "undefined") ? readability : {};

// Typing out "readability" really sucks.
window.rdb = readability;

$.extend(true, readability, {
	// TODO: This should really be handled by the templates
	APPEARANCE_URL: "/account/ajax/appearance/",

    /*
     * toggleBooleanField
     *
     * Allows a user to set an article's boolean fields
     *
     * @param {object} options : Used to pass the url to POST to and the callback function
    */
	toggleBooleanField: function(options){

        // Handle limited server error
        function _error_handler(XMLHttpRequest, textStatus, errorThrown){
            options.callback({
                success: false,
                article_id: options.article_id,
                request: XMLHttpRequest,
                status: textStatus,
                error: errorThrown
            });
        }

        // Handle server response, which could include an error!
        function _response_handler(data, textStatus, XMLHttpRequest){
			if(data){
				options.callback({
					success: data.success,
					article_id: options.article_id,
					value: data.value
				});
			}
			else {
				options.callback({
					success: false,
					article_id: options.article_id
				});
			}
        }

        $.ajax({
            type: "POST",
			url: options.url,
            dataType: "json",
            error: _error_handler,
			success: _response_handler
        });
	},

	/*
	 * setAppearanceValue
	 *
	 * Allows a user to set an appearance property to a specific value
	 *
	 * @param {string} model - The high level model to which the property belongs for determining the url
	 * @param {string} property - The name of the property to modify
	 * @param {string} value - The value the propety will be set to
	 * @param {object} callback
	*/
	setAppearanceValue: function(model, property, value, callback){

		function _error_handler(property, textStatus, errorThrown){
            callback({
                success: false,
                request: XMLHttpRequest,
                status: textStatus,
                error: errorThrown
            });
		}

        // Handle server response
        function _success_handler(data, textStatus){
			if(data){
				callback({
					success: data.success,
					property: data.property,
					value: data.value,
					previous_value: data.previous_value
				});
			}
			else {
				callback({
					success: false
				});
			}
        }

		$.ajax({
			type: 'POST',
			url: rdb.APPEARANCE_URL,
			data: {
				property: property,
				value: value
			},
			dataType: 'json',
            error: _error_handler,
            success: _success_handler
		});
	},

	/*
	 * BodyClass
	 *
	 * Allows us to fire callbacks based on body class modifications
	 * which gives us much greater control of what we end up doing
	 * with those signals.
	 *
	 * Usage:
	 *
	 * appearance_style_listener = readability.BodyClass.AddListener("appearance_*", function(event){
	 *		if( event.type == "Added" ){
	 *			// Do something when classes of appearance_* are added
	 *		}
	 *		else if( event.type == "Removed" ){
	 *			// Do something when they are removed
	 *		}
	 * })
	 *
	 * readability.BodyClass.addRemove("some_style","some_other_style");
	 *
	 * appearance_style_listener.remove()
	 * or
	 * readability.BodyClass.removeListener(appearance_style_listener);
	 *
	 */

	BodyClass: {
		$body: $('body'),
		listeners: {},
		ADDED: "added",
		REMOVED: "removed",

		addRemove: function(class_added, class_removed){

			var event = {
				class_added: class_added,
				class_removed: class_removed
			};

			if( class_added && ! this.$body.hasClass(class_added) ){
				this.$body.addClass(class_added);
				event.type = this.ADDED;
				fire_listeners(class_added, event);
			}
			if( class_removed && this.$body.hasClass(class_removed) ){
				this.$body.removeClass(class_removed);
				event.type = this.REMOVED;
				fire_listeners(class_removed, event);
			}

			function fire_listeners(class_name, event){
				for( var key in rdb.BodyClass.listeners ){
					if( rdb.BodyClass.listeners.hasOwnProperty(key) ){

						var listener = rdb.BodyClass.listeners[key];
							regex = new RegExp(listener.body_class);

						if( regex.test(class_name) ){
							listener.callback(event);
						}
					}
				}
			}

		},
		/*
		 * getClassByRegex returns the classname of a body class matching the regex or undefined
		 */
		getClassByRegex: function(regex){
			if(regex instanceof Object){
				var body_classes = this.$body.attr("class").split(" ");
				for(i=0; i<body_classes.length; i++){
					var body_class = body_classes[i];
					if(regex.test(body_class)){
						return body_class;
					}
				}
			}
			return undefined;
		},

		add: function(class_added){
			this.addRemove(class_added);
		},

		toggle: function(class_toggle){
			if( this.$body.hasClass(class_toggle) ){
				this.addRemove(undefined,class_toggle);
			}
			else {
				this.addRemove(class_toggle);
			}
		},

		remove: function(class_removed){
			this.addRemove(undefined, class_removed);
		},

		addListener: function(body_class, callback){
			var listener = {
				body_class: body_class,
				callback: callback,
				id: rdb.guid()
			};

			this.listeners[listener.id] = listener;

			return {
				remove: function(){
					delete rdb.BodyClass.listeners[listener.id];
				},
				listener: listener
			};
		},

		removeListener: function(listener_var){
			delete rdb.BodyClass.listeners[listener_var.listener.id];
		}
	},

	guid: function(){
		function S4() {
		   return (((1+Math.random())*0x10000)|0).toString(16).substring(1);
		}
	   return (S4()+S4()+"-"+S4()+"-"+S4()+"-"+S4()+"-"+S4()+S4()+S4());
	},

	// TODO figure out how to gracefully handle a maxed out localStorage
	local_storage_maxed: false,

	has_local_storage: function(){
		try {
			return 'localStorage' in window && window['localStorage'] !== null && rdb.local_storage_maxed !== true;
		}
		catch(e){
			return false;
		}
	},

	local_storage_get: function (prop) {
		if(rdb.has_local_storage()) {
			return localStorage.getItem(prop);
		}

		return null;
	},

	local_storage_set: function (prop, value) {
		if(rdb.has_local_storage()) {
			try {
				localStorage[prop] = value;
			} catch(e) {
				rdb.local_storage_maxed = true;
			}
		}
	},

	local_storage_delete: function (prop) {
		if(rdb.has_local_storage()) {
			delete localStorage[prop];
		}

		return null;
	}

});

// Users have a number of key:value settings that we can use to
// customize their experience. Use this to set each key.
// @param setting {obj}, key:value pair to set. Can only be one.
rdb.setKeyValue = function (setting, callback) {
    if (!$.isFunction(callback)) {
        callback = $.noop;
    }

    $.ajax({
        url: '/account/ajax/setkv',
        type: 'POST',
        data: JSON.stringify(setting),
        success: function (res) {
            callback(res);
        }
    });
};

// Convenience function to toggle profile boolean values on the UserProfile
// model.
rdb.setProfileBoolean = function (setting, callback) {
    if (!$.isFunction(callback)) {
        callback = $.noop;
    }

    $.ajax({
        url: '/account/ajax/setpbl',
        type: 'POST',
        data: JSON.stringify(setting),
        success: function (res) {
            callback(res);
        }
    });
};

// Get the size of the screen from the body:after CSS property
// return String
rdb.getScreenSize = function () {
    if (!window.getComputedStyle) {
        return null;
    }

    return window.getComputedStyle(document.body, ':after')
        .getPropertyValue('content')
        // Doing this for IE9. When it gets the string, it
        // includes " and ". Weird. Dumb.
        .replace(/\"/g, '');
};

// We want to collect as much data as we can about where our users are clicking.
// To start collecting stats on an element add the attribute data-stat and give
// it a descriptive value.
// This can also be called directly; rdb.stats.collect("stat-name");
rdb.stats = function () {
    var self = this;

    // If something fails with the request we don't want to hold anything else
    // up, so just call the callback if present.
    var successOrError = function () {
        if (self.callback) {
            self.callback();
        }
    };

    this.collect = function (stat, callback) {
        self.callback = callback;

        try {
            if (stat) {
                $.ajax({
                    type: 'POST',
                    url: '/ajax/s/i/',
                    data: {
                        'stat': stat
                    },
                    success: successOrError,
                    error: successOrError
                });
            }
        } catch (err) {
            successOrError();
        }
    };

    $('body').on('click', '[data-stat]', function (e) {
        var stat = $(e.target).attr('data-stat');

        if (!stat) {
            stat = $(e.currentTarget).attr('data-stat');
        }

        if (stat) {
            self.collect(stat);
        }
    });

    return this;
}();

// Polyfill for newer, html form features.
// Uses https://github.com/ryanseddon/H5F
// NOTE: After loading the Polyfill, you will need to run the H5F.setup method
// on the form.
rdb.h5f = {
    settings: {
        'validationMessages': {
            'valueMissing': 'Please fill out this field.',
            'typeMismatch': 'Please enter a valid email address.',
            'patternMismatch': 'Letters, numbers and underscores only please.'
        }
    },

    configure: function (options) {
        $.extend(true, this.settings, options);
    },

    hasSupport: function () {
        return !$('html').hasClass('no-formvalidation');
    },

    loadPolyfill: function (callback) {
        // We only want to load the form validation polyfill if we have to.
        // And only load it a single time.
        if (!this.hasSupport() && !$('body').hasClass('h5f-js')) {
            var h5fScript = document.createElement('script');
            h5fScript.src = '/media/js/libs/h5f/h5f.js';

            $('body').addClass('h5f-js').append(h5fScript);
        }

        if ($.isFunction(callback)) {
            callback();
        }
    },

    // There are times when we don't want to use the default validation messages.
    // This allows us to set a custom message based on the type of error
    // NOTE: This is a not fully complete polyfill
    getFieldValidationMessage: function (field) {
        var msgs = this.settings.validationMessages,
            validationMessage = null;

        // If the field has a title we'll use that has the error message,
        // if not look throught he validation Messages.
        if (field.title) {
            validationMessage = field.title;
        }
        else {
            var getKeyByValue = function (obj) {
                for (var prop in obj) {
                    if (obj[prop] === true) {
                        return prop;
                    }
                }
            };

            validationMessage = msgs[getKeyByValue(field.validity)] ||
                'Please check this field and try again.';
        }

        return validationMessage;
    }
};

// I moved this in from the previous readability.global.js. core.js is loaded
// on all the pages that global was, no need to have another HTTP request.
$(function() {
    // If this is chrome, we need to hijack the download button if it exists
    if($.browser.chrome) {
        $('.chrome-addon-install').on('click', function (ev) {
            ev.preventDefault();

            $.post('/addons/ajax/chrome/is/', {state: 'initiate'});

            chrome.webstore.install(
                "https://chrome.google.com/webstore/detail/oknpjjbmpnndlpmnhmekjpocelpnlfdi",
                // Success callback
                function () {
                    $.post('/addons/ajax/chrome/is/', {state: 'complate'});
                },
                // Error callback
                function (er) {
                    $.post('/addons/ajax/chrome/is/', {state: 'error', error: er});
                }
            );
        });
    }

    // Any links, like the bookmarklet itself, that should have their default actions overridden
    $('.rdb-no-link').click(function (e){ e.preventDefault(); });

    // HTML5 Placeholder Deprecation JS.
    // Automatically support the placeholder attribute in browsers that don't support it.
    if (!('placeholder' in document.createElement('input'))) {
        var showPlaceholder = function () {
            var $this = $(this),
                placeholder = $this.attr('placeholder');

            if ($this.val() === '' && placeholder) {
                $this.val(placeholder).addClass('placeholderActive');
            }
        };

        var hidePlaceholder = function () {
            var $this = $(this),
                placeholder = $this.attr('placeholder');

            if (placeholder && $this.val() == placeholder) {
                $this.val('').removeClass('placeholderActive');
            }
        };

        // Look for forms with inputs and/or textareas with a placeholder attribute in them
        $('form').submit(function () {
            // Clear the placeholder values so they don't get submitted
            $('.placeholderActive', this).val('');
        });

        // Clear placeholder values upon page reload
        $(window).unload(function () {
            $('.placeholderActive').val('');
        });

        $(':input').not('input[type=password]').each(showPlaceholder)
            .blur(showPlaceholder).focus(hidePlaceholder);
    }

    // Show navigation on small screens
    $('a.show-navigation').click(function (e) {
        e.preventDefault();

        $('.rdb-menu').toggleClass('small-screen-hide');
    });

    try {
        if (window.Modernizr) {
            // Browsers that have no media query support
            // NOTE: This media query is just a general catch all to make IE9 not
            // be a little bitch and return false when using an empty string.
            if (!Modernizr.mq('only all and (min-width: 0px)')) {
                $('html').addClass('no-mq');
            } else {
                // We only want to hide the menu with this class if the device
                // has JS enabled and supports media queries
                $('#rdb-menu').addClass('small-screen-hide');
            }
        }
    }
    catch (e) {
        console.log(e);
    }

    // If the page has colorbox, set some defaults and init
    // on links that need it.
    if ("colorbox" in window.jQuery.fn) {
        $.extend($.colorbox.settings, {
            transition: 'none',
            opacity: 1,
            speed: 0
        });

        $("a[rel=colorbox]").colorbox({
            rel: 'nofollow'
        });

        // NOTE: From here down isn't really the best place for this

        // Apps page, Add-ons learn more link
        $('a[href=#colorbox-addons]').colorbox({
            inline : true,
            width: '640px'
        });

        // Learn More Kindle video
        $("a[href='#kindle-video-demo']").colorbox({
            html: '<div class="cboxContentWrapper cboxVideoModal"><iframe src="http://player.vimeo.com/video/30451002?title=0&amp;byline=0&amp;portrait=0" width="620" height="349" frameborder="0" webkitAllowFullScreen allowFullScreen></iframe></div>'
        });
    }

    // Open a modal that gives a download option for the logos when
    // right clicking on the logo
    $('[role=banner]').on('contextmenu', '#logo', function (e) {
        e.preventDefault();
        var el = $(_.template($('#logo-download').html(), {}));

        // We kept seeing weirdness with the template not being fully loaded
        // so the lightbox would have a scrollbar, wait for it to be ready.
        setTimeout(function () {
            $.colorbox({
                html: el
            });
        }, 200);
    });
});

