package search.eightpuzzle;

import java.util.HashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.PriorityQueue;

public
class
AStar
{

  private
  final
  PuzzleState
  goal;

  private
  final
  Heuristic
  heuristic;

  private
  final
  PriorityQueue<WeightedState>
  open;

  /**
   * This is a mapping from a state (its unique hashcode) to the lowest cost we
   * found for this state. So it is the CLOSED list, but also stores the costs
   * for nodes on OPEN.
   */
  private
  final
  Map<Integer, Integer> minCosts;

  public
  AStar(
      final PuzzleState start,
      final PuzzleState goal,
      final Heuristic heuristic)
  {
    assert start != null;
    assert goal != null;
    assert heuristic != null;

    this.goal = goal;
    this.heuristic = heuristic;
    this.open = new PriorityQueue<AStar.WeightedState>();
    this.open.add(new WeightedState(0, start));
    this.minCosts = new HashMap<Integer, Integer>();
    this.minCosts.put(start.hashCode(), 0);
  }

  private
  PuzzleState
  search()
  {
    PuzzleState state = null;
    int counter = 0;
    while (state == null)
    {
      ++counter; // Number of expanded nodes
      try
      {
        state = this.advance(); // Expand next on open
      }
      catch (NoSuchElementException e)
      {
        // No node left on open
        System.out.println("Aborted after " + counter + " steps");
        return null;
      }
    }
    System.out.println("Found goal after " + counter + " steps");
    return state;
  }

  private
  PuzzleState
  advance()
  {
    final WeightedState n = this.nextStateFromOpen();
    if (n == null)
    {
      throw new NoSuchElementException(); // OPEN emtpy: FAIL
    }
//    System.out.println("Checking " + n.state + " with cost " + n.currentCost + " and total estimate " + n.estimatedCost);
    if (n.state.equals(this.goal))
    {
      return n.state; // found goal
    }

    for (PuzzleState successor : n.state) // Expand n
    {
      final int newCost = n.currentCost + 1; // We have always a cost of 1
      final Integer oldCost = this.minCosts.get(successor.hashCode());
      if (oldCost == null || newCost < oldCost.intValue())
      {
        this.open.add(new WeightedState(newCost, successor));
        this.minCosts.put(successor.hashCode(), newCost);
        // Note that I do not delete the old state from OPEN but then ignore
        // it later on. (in nextStateFromOpen()). Costs more memory, but safes
        // time.
      }
    }
    return null;
  }

  private
  WeightedState
  nextStateFromOpen()
  {
    WeightedState n = this.open.poll();
    if (n == null) { return null; }  // Open is empty
    while (n.currentCost > this.minCosts.get(n.state.hashCode()))
    {
      // A smaller path had been found and used...
      // Note: in A* this would have been removed from OPEN when the smaller
      // path was found. In my implementation it is not deleted, but ignored
      // here.
      n = this.open.poll();
      if (n == null) { return null; }
    }
    return n;
  }

  /**
   * A state including currentCost (g) and estimatedCost (f). The cost is
   * estimated in the constructor. The estimatedCost is used for sorting the
   * states in OPEN.
   */
  private
  class
  WeightedState
  implements Comparable<WeightedState>
  {

    private
    final
    int
    currentCost;

    private
    final
    int
    estimatedCost;

    private
    final
    PuzzleState state;

    private
    WeightedState(
        final int currentCost,
        final PuzzleState state)
    {
      assert state != null;
      this.currentCost = currentCost;
      this.state = state;
      this.estimatedCost = this.currentCost +
          AStar.this.heuristic.estimate(this.state, AStar.this.goal);
    }

    @Override
    public
    int
    compareTo(
        WeightedState o)
    {
      // Compares this object with the specified object for order.
      // Returns a negative integer, zero, or a positive integer as this object
      // is less than, equal to, or greater than the specified object.
      final int weightDifference = this.estimatedCost - o.estimatedCost;
      return weightDifference;
    }
    
  }

  private
  static
  void
  run(
      final PuzzleState start,
      final PuzzleState goal,
      final Heuristic heuristic)
  {
    System.out.println();
    final AStar search = new AStar(start, goal, heuristic);
    System.out.println("Start " + start + " with heuristic " + heuristic);
    final PuzzleState found = search.search();
    if (found != null)
    {
      System.out.println(found.printHistory());
      System.out.println("Goal " + goal + " reached in "
          + found.getDepth() + " moves.");
    }
    else
    {
      System.out.println("Goal " + goal + " not reached ");
    }
  }

  public static void main(String[] args)
  {
    final PuzzleState start = new PuzzleState(6, 4, 7, 8, 5, 0, 3, 2, 1);
    final PuzzleState goal = new PuzzleState(1, 2, 3, 4, 5, 6, 7, 8, 0);
    AStar.run(start, goal, new ZeroHeuristic());
    AStar.run(start, goal, new MismatchHeuristic());
    AStar.run(start, goal, new ManhattanHeuristic());
  }

}
