%% Tutorial_UQ_Reliability: Model Definition, Uncertainty Quantification and Reliability Analysis
% In this tutorial I will show you how to set-up and run a reliability
% anylisis using cossan. Three different reliability algorithms will be
% used and compared.


% Reset the random number generator in order to obtain always the same results.
% DO NOT CHANGE THE VALUES OF THE SEED
OpenCossan.resetRandomNumberGenerator(51125);

%% Preparation of the Input
% In this sesion we will define the basic input quantities of the model.
% The basic input classes available in cossan are Random Variables,
% Parameter and Function

%% Definition of the Parameters
% Constant parameters of the model
L=Parameter('Sdescription','Beam Length','value',1.8);
b=Parameter('Sdescription','Beam Width','value',0.12);
% Parameter for the Maximum displacement (used in the Reliability Analysis)
maxDiplacement=Parameter('Sdescription','Maximum allowed displacement','value',0.015);

%% Definition of the Random Varibles
% Random parameters of the model
P=RandomVariable('Sdescription','Load',...
                 'Sdistribution','lognormal','mean',5000,'std',400);
h=RandomVariable('Sdescription','Beam Heigth',...
                 'Sdistribution','normal','mean',0.24,'std',0.01);
rho=RandomVariable('Sdescription','density',...
                   'Sdistribution','lognormal','mean',600,'std',140);
E=RandomVariable('Sdescription','Young''s modulus',...
                 'Sdistribution','lognormal','mean',10e9,'std',1.6e9);

%% Definition of the Function
% The geometric moment of inertia is a basic function of the section width
% and height
I=Function('Sdescription','Moment of Inertia','Sexpression','<&b&>.*<&h&>.^3/12');

%% Random variable set definition
% Create the set of Random Variables (Random cariables can be correlated)
Xrvset=RandomVariableSet('CXrandomVariables',{P h rho E},...
                         'CSmembers',{'P' 'h' 'rho' 'E'});

% If you want to have a correlated set of RVs
Mcorrelation=eye(4);
Mcorrelation(3,4)=0.8; % Add correlation between rho and E
Mcorrelation(4,3)=0.8;
Xrvset_correlated=RandomVariableSet('CXrandomVariables',{P h rho E},...
                         'CSmembers',{'P' 'h' 'rho' 'E'},...
                         'Mcorrelation',Mcorrelation);

%% Prepare Input Object
% The objects constructed above are collected into an Input Object
Xinput=Input('CXmembers',{L b Xrvset I maxDiplacement},...
             'CSmembers',{'L' 'b' 'Xrvset' 'I' 'maxDiplacement'});
% Show summary of the Input Object
display(Xinput)
%% Preparation of the Evaluator
% The evaluator is a box that contains all the solvers. It is totally
% independent from the probabilitstsit definition of the inputs! It also
% have the information about the order of the solver and the dependencies
% between I/O of each solver. 
% Possible solvers availabe (at the moment) are Matlab scripts and
% functions, as well as standalone solvers (as long as ASCII files are used
% for input and output - more in the next example).

% Use of a matlab script to compute the Beam displacement
Xmio=Mio('Spath',pwd,'Sfile','tipDisp_matrix.m',...
         'Cinputnames',{'I' 'b' 'L' 'h' 'rho' 'P' 'E'}, ...
         'Coutputnames',{'w'},...
         'Lfunction',false,...
         'Liomatrix',true,'Liostructure',false); % Possible I/O format are matrix, table, structure
     
% Add the MIO object to an Evaluator object
Xevaluator=Evaluator('CXmembers',{Xmio},'CSmembers',{'Xmio'});

%% Preparation of the Model
% A model unites the black-box solvers together with the probabilistic
% inputs.

% Define the Physical Model
XmodelBeamMatlab=Model('Xinput',Xinput,'Xevaluator',Xevaluator);

% Perform deterministic analysis
Xout=XmodelBeamMatlab.deterministicAnalysis;
NominalDisplacement=Xout.getValues('Sname','w');


%% Uncertainty Quantification
% You can run a Model with randomly sampled values of the probabilistic
% inputs to propagate uncertainties. Other UQ tools available are 
% sensitivity (local and global), extreme case analysis, and metamodel
% training.

% Define simulation method
Xmc=MonteCarlo('Nsamples',1000);
% preform Analysis
XsimOutMC=Xmc.apply(XmodelBeamMatlab);

%% Plot Results
% show scatter of the beam tip displacement
f1=figure;
Vw=XsimOutMC.getValues('Sname','w');
hist(Vw,50); 
xlabel('Tip displacement'); ylabel('Frequency');

f2=figure;
scatter([XsimOutMC.Tvalues.P],[XsimOutMC.Tvalues.w]);
xlabel('Tip load'); ylabel('Tip displacement');

%% RELIABILITY ANALYSIS 
% The reliaility analysis is performed in this tutorial using standard
% Monte Carlo, Line Sampling and Subset Simulation. The results with
% different nr. of samples are compared.

%% Define a Probabilistic Model
% The Probabilistic Model is the union of the performance function g 
% (g<0 failure, g>0 safe)

% Performance Function
Xperfun = PerformanceFunction('Sdemand','w','Scapacity','maxDiplacement','Soutputname','g');
% Define a Probabilistic Model
XprobModelBeamMatlab=ProbabilisticModel('Xmodel',XmodelBeamMatlab,'XperformanceFunction',Xperfun);

%% Reliability Analysis via Monte Carlo Sampling
% The Monte Carlo simulation is used here to estimate the failure probability

% Compute Reference Solution
Xmc=MonteCarlo('Nsamples',5e5,'Nbatches',1);

% Run Reliability Analysis
[XfailureProbMC, XoutMC]=Xmc.computeFailureProbability(XprobModelBeamMatlab);
% Show the estimated failure probability
display(XfailureProbMC);


%% Reliability Analysis via LineSampling
% Line Sampling requires the definition of the so-called important direction.
% It can be computed usig the sensitivity method. For instance here the Local
% Sensitivity Analysis is computed.

XlsFD=LocalSensitivityFiniteDifference('Xmodel',XprobModelBeamMatlab,'Coutputname',{'Vg'});
display(XlsFD)

% Compute the LocalSensitivityMeasure
XlocalSensitivity = XlsFD.computeIndices;

%%
% Use sensitivity measures computed previously to define the important
% direction for the LineSampling and compute the probability of failure.
XLS=LineSampling('XlocalSensitivityMeasures',XlocalSensitivity,'Nlines',50);
% Run Analysis
[XfailureProbLS, XoutLS]=XLS.computeFailureProbability(XprobModelBeamMatlab);
% Show Results
display(XfailureProbLS);
display(XoutLS);

%% Some informative plots from Line Sampling 
f3=XoutLS.plotLines;

%% Reliability Analysis via SubSet Simulation
% Subset simulation computes the failure probability as the product of
% intermediate dependent failure probabilities.

XSS = SubSet('NinitialSamples',500,'Nsamples',50,...
    'targetfailureprobability',1e-1,...
    'Nmaxlevels',10);

%%
% Run Analysis
[XfailureProbSS, XoutSS]=XSS.computeFailureProbability(XprobModelBeamMatlab);
% Show Results
display(XfailureProbSS);
%% Some informative plot from subset 
f4=XoutSS.plotMarkovChains('Cnames',{'P','h'});